package main

import (
	"bufio"
	"fmt"
	"io"
	"log"
	"net"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/gdamore/tcell/v2"
)

type State struct {
	mutex sync.Mutex
	data  map[string]string
}

var gState State

func init() {
	gState.data = make(map[string]string)
}

func run() {
	if gLogPath != "" {
		f, err := os.OpenFile(gLogPath, os.O_RDWR|os.O_CREATE|os.O_APPEND, 0o600)
		if err != nil {
			log.Fatalf("failed to open log file: %s", err)
		}
		defer f.Close()
		log.SetOutput(f)
	} else {
		log.SetOutput(io.Discard)
	}

	log.Printf("*************** starting client, PID: %d ***************", os.Getpid())

	var screen tcell.Screen
	var err error
	if screen, err = tcell.NewScreen(); err != nil {
		log.Fatalf("creating screen: %s", err)
	} else if err = screen.Init(); err != nil {
		log.Fatalf("initializing screen: %s", err)
	}
	if gOpts.mouse {
		screen.EnableMouse()
	}
	screen.EnablePaste()

	ui := newUI(screen)
	nav := newNav(ui)
	app := newApp(ui, nav)

	if err := nav.sync(); err != nil {
		app.ui.echoerrf("sync: %s", err)
	}

	if err := app.readHistory(); err != nil {
		app.ui.echoerrf("reading history file: %s", err)
	}

	app.loop()

	app.ui.screen.Fini()

	if gLastDirPath != "" {
		writeLastDir(gLastDirPath, app.nav.currDir().path)
	}

	if gSelectionPath != "" && len(app.selectionOut) > 0 {
		writeSelection(gSelectionPath, app.selectionOut)
	}

	if gPrintLastDir {
		fmt.Println(app.nav.currDir().path)
	}

	if gPrintSelection && len(app.selectionOut) > 0 {
		for _, file := range app.selectionOut {
			fmt.Println(file)
		}
	}
}

func writeLastDir(filename string, lastDir string) {
	f, err := os.Create(filename)
	if err != nil {
		log.Printf("opening last dir file: %s", err)
		return
	}
	defer f.Close()

	_, err = f.WriteString(lastDir)
	if err != nil {
		log.Printf("writing last dir file: %s", err)
	}
}

func writeSelection(filename string, selection []string) {
	f, err := os.Create(filename)
	if err != nil {
		log.Printf("opening selection file: %s", err)
		return
	}
	defer f.Close()

	_, err = f.WriteString(strings.Join(selection, "\n"))
	if err != nil {
		log.Printf("writing selection file: %s", err)
	}
}

func readExpr() <-chan expr {
	ch := make(chan expr)

	go func() {
		duration := 100 * time.Millisecond

		c, err := net.Dial(gSocketProt, gSocketPath)
		for err != nil {
			log.Printf("connecting server: %s", err)
			time.Sleep(duration)
			duration *= 2
			c, err = net.Dial(gSocketProt, gSocketPath)
		}

		if _, err := fmt.Fprintf(c, "conn %d\n", gClientID); err != nil {
			log.Fatalf("registering with server: %s", err)
		}

		ch <- &callExpr{"sync", nil, 1}
		ch <- &callExpr{"on-init", nil, 1}

		s := bufio.NewScanner(c)
		for s.Scan() {
			log.Printf("recv: %s", s.Text())

			// `query` has to be handled outside of the main thread, which is
			// blocked when running a synchronous shell command ("$" or "!").
			// This is important since `query` is often the result of the user
			// running `$lf -remote "query $id <something>"`.
			if word, rest := splitWord(s.Text()); word == "query" {
				gState.mutex.Lock()
				state := gState.data[rest]
				gState.mutex.Unlock()
				if _, err := fmt.Fprintln(c, state); err != nil {
					log.Fatalf("sending response to server: %s", err)
				}
			} else {
				p := newParser(strings.NewReader(s.Text()))
				if p.parse() {
					ch <- p.expr
				}
			}
		}

		if err := s.Err(); err != nil {
			log.Printf("reading from server: %s", err)
		}

		c.Close()
	}()

	return ch
}

func remote(cmd string) error {
	c, err := net.Dial(gSocketProt, gSocketPath)
	if err != nil {
		return fmt.Errorf("dialing to send server: %w", err)
	}

	if _, err := fmt.Fprintln(c, cmd); err != nil {
		return fmt.Errorf("sending command to server: %w", err)
	}

	// XXX: Standard net.Conn interface does not include a CloseWrite method
	// but net.UnixConn and net.TCPConn implement it so the following should be
	// safe as long as we do not use other types of connections. We need
	// CloseWrite to notify the server that this is not a persistent connection
	// and it should be closed after the response.
	switch c := c.(type) {
	case *net.TCPConn:
		c.CloseWrite()
	case *net.UnixConn:
		c.CloseWrite()
	}

	// The most straightforward way to write the response to stdout would be
	// to do "io.Copy(os.Stdout, c)". However, on Linux, if stdout is connected to
	// something like a pager that isn't immediately reading everything,
	// this runs into Go bug https://github.com/golang/go/issues/75304 and
	// busy-loops, eating a whole CPU.
	response, err := io.ReadAll(c)
	if err != nil {
		return fmt.Errorf("reading response from server: %w", err)
	}
	c.Close()
	os.Stdout.Write(response)

	return nil
}
