// This is free and unencumbered software released into the public
// domain.  For more information, see <http://unlicense.org> or the
// accompanying UNLICENSE file.

package mocks_test

import (
	"bytes"
	"go/ast"
	"go/format"
	"testing"

	"github.com/a8m/expect"
	"github.com/nelsam/hel/v2/mocks"
	"github.com/nelsam/hel/v2/types"
)

func TestGenerate(t *testing.T) {
	expect := expect.New(t)

	types := []*ast.TypeSpec{
		typeSpec(expect, `
  type Foo interface {
   Bar() int
  }`),
		typeSpec(expect, `
  type Bar interface {
   Foo(foo string)
   Baz()
  }`),
	}

	mockFinder := newMockTypeFinder()
	close(mockFinder.DependenciesOutput.Dependencies)
	mockFinder.ExportedTypesOutput.Types <- types
	m, err := mocks.Generate(mockFinder)
	expect(err).To.Be.Nil().Else.FailNow()
	expect(m).To.Have.Len(2).Else.FailNow()
	expect(m[0]).To.Equal(mockFor(expect, types[0]))
	expect(m[1]).To.Equal(mockFor(expect, types[1]))
}

func TestOutput(t *testing.T) {
	expect := expect.New(t)

	types := []*ast.TypeSpec{
		typeSpec(expect, `
  type Foo interface {
   Bar() int
  }`),
		typeSpec(expect, `
  type Bar interface {
   Foo(foo string) Foo
   Baz()
   Bacon(func(Eggs) Eggs) func(Eggs) Eggs
  }`),
	}

	mockFinder := newMockTypeFinder()
	close(mockFinder.DependenciesOutput.Dependencies)
	mockFinder.ExportedTypesOutput.Types <- types
	m, err := mocks.Generate(mockFinder)
	expect(err).To.Be.Nil().Else.FailNow()

	buf := bytes.Buffer{}
	m.Output("foo", "test/withoutimports", 100, &buf)

	expected, err := format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo

 type mockFoo struct {
  BarCalled chan bool
  BarOutput struct {
   Ret0 chan int
  }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.BarCalled = make(chan bool, 100)
  m.BarOutput.Ret0 = make(chan int, 100)
  return m
 }
 func (m *mockFoo) Bar() int {
  m.BarCalled <- true
  return <-m.BarOutput.Ret0
 }

 type mockBar struct {
  FooCalled chan bool
  FooInput struct {
   Foo chan string
  }
  FooOutput struct {
   Ret0 chan Foo
  }
  BazCalled chan bool
  BaconCalled chan bool
  BaconInput struct {
    Arg0 chan func(Eggs) Eggs
  }
  BaconOutput struct {
    Ret0 chan func(Eggs) Eggs
  }
 }

 func newMockBar() *mockBar {
  m := &mockBar{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.Foo = make(chan string, 100)
  m.FooOutput.Ret0 = make(chan Foo, 100)
  m.BazCalled = make(chan bool, 100)
  m.BaconCalled = make(chan bool, 100)
  m.BaconInput.Arg0 = make(chan func(Eggs) Eggs, 100)
  m.BaconOutput.Ret0 = make(chan func(Eggs) Eggs, 100)
  return m
 }
 func (m *mockBar) Foo(foo string) Foo {
  m.FooCalled <- true
  m.FooInput.Foo <- foo
  return <-m.FooOutput.Ret0
 }
 func (m *mockBar) Baz() {
  m.BazCalled <- true
 }
 func (m *mockBar) Bacon(arg0 func(Eggs) Eggs) func(Eggs) Eggs {
  m.BaconCalled <- true
  m.BaconInput.Arg0 <- arg0
  return <-m.BaconOutput.Ret0
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))

	m.PrependLocalPackage("foo")
	buf = bytes.Buffer{}
	m.Output("foo_test", "test/withoutimports", 100, &buf)

	expected, err = format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo_test

 type mockFoo struct {
  BarCalled chan bool
  BarOutput struct {
   Ret0 chan int
  }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.BarCalled = make(chan bool, 100)
  m.BarOutput.Ret0 = make(chan int, 100)
  return m
 }
 func (m *mockFoo) Bar() int {
  m.BarCalled <- true
  return <-m.BarOutput.Ret0
 }

 type mockBar struct {
  FooCalled chan bool
  FooInput struct {
   Foo chan string
  }
  FooOutput struct {
   Ret0 chan foo.Foo
  }
  BazCalled chan bool
  BaconCalled chan bool
  BaconInput struct {
    Arg0 chan func(foo.Eggs) foo.Eggs
  }
  BaconOutput struct {
    Ret0 chan func(foo.Eggs) foo.Eggs
  }
 }

 func newMockBar() *mockBar {
  m := &mockBar{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.Foo = make(chan string, 100)
  m.FooOutput.Ret0 = make(chan foo.Foo, 100)
  m.BazCalled = make(chan bool, 100)
  m.BaconCalled = make(chan bool, 100)
  m.BaconInput.Arg0 = make(chan func(foo.Eggs) foo.Eggs, 100)
  m.BaconOutput.Ret0 = make(chan func(foo.Eggs) foo.Eggs, 100)
  return m
 }
 func (m *mockBar) Foo(foo string) foo.Foo {
  m.FooCalled <- true
  m.FooInput.Foo <- foo
  return <-m.FooOutput.Ret0
 }
 func (m *mockBar) Baz() {
  m.BazCalled <- true
 }
 func (m *mockBar) Bacon(arg0 func(foo.Eggs) foo.Eggs) func(foo.Eggs) foo.Eggs {
  m.BaconCalled <- true
  m.BaconInput.Arg0 <- arg0
  return <-m.BaconOutput.Ret0
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))

	m.SetBlockingReturn(true)
	buf = bytes.Buffer{}
	m.Output("foo_test", "test/withoutimports", 100, &buf)

	expected, err = format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo_test

 type mockFoo struct {
  BarCalled chan bool
  BarOutput struct {
   Ret0 chan int
  }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.BarCalled = make(chan bool, 100)
  m.BarOutput.Ret0 = make(chan int, 100)
  return m
 }
 func (m *mockFoo) Bar() int {
  m.BarCalled <- true
  return <-m.BarOutput.Ret0
 }

 type mockBar struct {
  FooCalled chan bool
  FooInput struct {
   Foo chan string
  }
  FooOutput struct {
   Ret0 chan foo.Foo
  }
  BazCalled chan bool
  BazOutput struct {
   BlockReturn chan bool
  }
  BaconCalled chan bool
  BaconInput struct {
    Arg0 chan func(foo.Eggs) foo.Eggs
  }
  BaconOutput struct {
    Ret0 chan func(foo.Eggs) foo.Eggs
  }
 }

 func newMockBar() *mockBar {
  m := &mockBar{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.Foo = make(chan string, 100)
  m.FooOutput.Ret0 = make(chan foo.Foo, 100)
  m.BazCalled = make(chan bool, 100)
  m.BazOutput.BlockReturn = make(chan bool, 100)
  m.BaconCalled = make(chan bool, 100)
  m.BaconInput.Arg0 = make(chan func(foo.Eggs) foo.Eggs, 100)
  m.BaconOutput.Ret0 = make(chan func(foo.Eggs) foo.Eggs, 100)
  return m
 }
 func (m *mockBar) Foo(foo string) foo.Foo {
  m.FooCalled <- true
  m.FooInput.Foo <- foo
  return <-m.FooOutput.Ret0
 }
 func (m *mockBar) Baz() {
  m.BazCalled <- true
  <-m.BazOutput.BlockReturn
 }
 func (m *mockBar) Bacon(arg0 func(foo.Eggs) foo.Eggs) func(foo.Eggs) foo.Eggs {
  m.BaconCalled <- true
  m.BaconInput.Arg0 <- arg0
  return <-m.BaconOutput.Ret0
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))
}

func TestOutput_Dependencies(t *testing.T) {
	expect := expect.New(t)

	typs := []*ast.TypeSpec{
		typeSpec(expect, `
  type Bar interface {
    Foo(foo string) (Foo, b.Foo)
  }`),
		typeSpec(expect, `
  type Foo interface {
    Foo() string
  }`),
	}

	barDeps := []types.Dependency{
		// We shouldn't see duplicates of types we're
		// already mocking.
		{
			Type: typeSpec(expect, `
			type Foo interface{
				Foo() string
			}`),
		},
		// Different package names should have the type
		// name altered.
		{
			PkgName: "b",
			Type: typeSpec(expect, `
			type Foo interface{
				Foo() string
			}`),
		},
		// Different types entirely should be supported.
		{
			PkgPath: "some/path/to/foo",
			PkgName: "baz",
			Type: typeSpec(expect, `
			type Baz interface {
				Baz() Baz
			}
			`),
		},
	}
	fooDeps := []types.Dependency{
		// We shouldn't see duplicate dependencies from
		// previous types, either.
		{
			PkgPath: "some/path/to/foo",
			PkgName: "baz",
			Type: typeSpec(expect, `
			type Baz interface {
				Baz() Baz
			}
			`),
		},
	}

	mockFinder := newMockTypeFinder()
	mockFinder.ExportedTypesOutput.Types <- typs
	mockFinder.DependenciesOutput.Dependencies <- barDeps
	mockFinder.DependenciesOutput.Dependencies <- fooDeps
	m, err := mocks.Generate(mockFinder)
	expect(err).To.Be.Nil().Else.FailNow()

	buf := bytes.Buffer{}
	m.Output("foo", "test/withoutimports", 100, &buf)

	// TODO: For some reason, functions are coming out without
	// whitespace between them.  We need to figure that out.
	expected, err := format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo

 type mockBar struct {
  FooCalled chan bool
  FooInput struct {
    Foo chan string
  }
  FooOutput struct {
    Ret0 chan Foo
    Ret1 chan b.Foo
  }
 }

 func newMockBar() *mockBar {
  m := &mockBar{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.Foo = make(chan string, 100)
  m.FooOutput.Ret0 = make(chan Foo, 100)
  m.FooOutput.Ret1 = make(chan b.Foo, 100)
  return m
 }
 func (m *mockBar) Foo(foo string) (Foo, b.Foo) {
  m.FooCalled <- true
  m.FooInput.Foo <- foo
  return <-m.FooOutput.Ret0, <-m.FooOutput.Ret1
 }

 type mockFoo struct {
  FooCalled chan bool
  FooOutput struct {
   Ret0 chan string
  }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.FooCalled = make(chan bool, 100)
  m.FooOutput.Ret0 = make(chan string, 100)
  return m
 }
 func (m *mockFoo) Foo() string {
  m.FooCalled <- true
  return <-m.FooOutput.Ret0
 }

 type mockBFoo struct {
  FooCalled chan bool
  FooOutput struct {
   Ret0 chan string
  }
 }

 func newMockBFoo() *mockBFoo {
  m := &mockBFoo{}
  m.FooCalled = make(chan bool, 100)
  m.FooOutput.Ret0 = make(chan string, 100)
  return m
 }
 func (m *mockBFoo) Foo() string {
  m.FooCalled <- true
  return <-m.FooOutput.Ret0
 }

 type mockBaz struct {
  BazCalled chan bool
  BazOutput struct {
   Ret0 chan baz.Baz
  }
 }

 func newMockBaz() *mockBaz {
  m := &mockBaz{}
  m.BazCalled = make(chan bool, 100)
  m.BazOutput.Ret0 = make(chan baz.Baz, 100)
  return m
 }
 func (m *mockBaz) Baz() baz.Baz {
  m.BazCalled <- true
  return <-m.BazOutput.Ret0
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))
}

func TestOutputWithPackageInputs(t *testing.T) {
	expect := expect.New(t)

	types := []*ast.TypeSpec{
		typeSpec(expect, `
  type Foo interface {
   Bar() int
  }`),
		typeSpec(expect, `
  type Bar interface {
   Foo(foo string) Foo
   Baz()
   Bacon(func(Eggs) Eggs) func(Eggs) Eggs
  }`),
	}

	mockFinder := newMockTypeFinder()
	close(mockFinder.DependenciesOutput.Dependencies)
	mockFinder.ExportedTypesOutput.Types <- types
	m, err := mocks.Generate(mockFinder)
	expect(err).To.Be.Nil().Else.FailNow()

	buf := bytes.Buffer{}
	m.Output("foo", "test/withimports", 100, &buf)

	expected, err := format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo

 import (
	thisIsFmt "fmt"
	"strconv"
 )

 type mockFoo struct {
  BarCalled chan bool
  BarOutput struct {
   Ret0 chan int
  }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.BarCalled = make(chan bool, 100)
  m.BarOutput.Ret0 = make(chan int, 100)
  return m
 }
 func (m *mockFoo) Bar() int {
  m.BarCalled <- true
  return <-m.BarOutput.Ret0
 }

 type mockBar struct {
  FooCalled chan bool
  FooInput struct {
   Foo chan string
  }
  FooOutput struct {
   Ret0 chan Foo
  }
  BazCalled chan bool
  BaconCalled chan bool
  BaconInput struct {
    Arg0 chan func(Eggs) Eggs
  }
  BaconOutput struct {
    Ret0 chan func(Eggs) Eggs
  }
 }

 func newMockBar() *mockBar {
  m := &mockBar{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.Foo = make(chan string, 100)
  m.FooOutput.Ret0 = make(chan Foo, 100)
  m.BazCalled = make(chan bool, 100)
  m.BaconCalled = make(chan bool, 100)
  m.BaconInput.Arg0 = make(chan func(Eggs) Eggs, 100)
  m.BaconOutput.Ret0 = make(chan func(Eggs) Eggs, 100)
  return m
 }
 func (m *mockBar) Foo(foo string) Foo {
  m.FooCalled <- true
  m.FooInput.Foo <- foo
  return <-m.FooOutput.Ret0
 }
 func (m *mockBar) Baz() {
  m.BazCalled <- true
 }
 func (m *mockBar) Bacon(arg0 func(Eggs) Eggs) func(Eggs) Eggs {
  m.BaconCalled <- true
  m.BaconInput.Arg0 <- arg0
  return <-m.BaconOutput.Ret0
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))
}

func TestOutput_ReceiverNameInArgs(t *testing.T) {
	expect := expect.New(t)

	types := []*ast.TypeSpec{
		typeSpec(expect, `
  type Foo interface {
   Foo(m string)
  }`),
	}

	mockFinder := newMockTypeFinder()
	close(mockFinder.DependenciesOutput.Dependencies)
	mockFinder.ExportedTypesOutput.Types <- types
	m, err := mocks.Generate(mockFinder)
	expect(err).To.Be.Nil().Else.FailNow()

	buf := bytes.Buffer{}
	m.Output("foo", "test/withoutimports", 100, &buf)

	expected, err := format.Source([]byte(`
 // This file was generated by github.com/nelsam/hel.  Do not
 // edit this code by hand unless you *really* know what you're
 // doing.  Expect any changes made manually to be overwritten
 // the next time hel regenerates this file.

 package foo

 type mockFoo struct {
   FooCalled chan bool
   FooInput struct {
     M chan string
   }
 }

 func newMockFoo() *mockFoo {
  m := &mockFoo{}
  m.FooCalled = make(chan bool, 100)
  m.FooInput.M = make(chan string, 100)
  return m
 }
 func (m *mockFoo) Foo(m_ string) {
  m.FooCalled <- true
  m.FooInput.M <- m_
 }
 `))
	expect(err).To.Be.Nil().Else.FailNow()
	expect(buf.String()).To.Equal(string(expected))
}

func mockFor(expect func(interface{}) *expect.Expect, spec *ast.TypeSpec) mocks.Mock {
	m, err := mocks.For(spec)
	expect(err).To.Be.Nil()
	return m
}
