/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.messaging;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.firebase.ErrorCode;
import com.google.firebase.FirebaseApp;
import com.google.firebase.ImplFirebaseTrampolines;
import com.google.firebase.internal.CallableOperation;
import com.google.firebase.internal.FirebaseService;
import com.google.firebase.internal.NonNull;
import com.google.firebase.messaging.BatchResponse;
import com.google.firebase.messaging.BatchResponseImpl;
import com.google.firebase.messaging.FirebaseMessagingClient;
import com.google.firebase.messaging.FirebaseMessagingClientImpl;
import com.google.firebase.messaging.FirebaseMessagingException;
import com.google.firebase.messaging.InstanceIdClient;
import com.google.firebase.messaging.InstanceIdClientImpl;
import com.google.firebase.messaging.Message;
import com.google.firebase.messaging.MulticastMessage;
import com.google.firebase.messaging.SendResponse;
import com.google.firebase.messaging.TopicManagementResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;

public class FirebaseMessaging {
    private final FirebaseApp app;
    private final Supplier<? extends FirebaseMessagingClient> messagingClient;
    private final Supplier<? extends InstanceIdClient> instanceIdClient;
    private static final String SERVICE_ID = FirebaseMessaging.class.getName();

    private FirebaseMessaging(Builder builder) {
        this.app = (FirebaseApp)Preconditions.checkNotNull((Object)builder.firebaseApp);
        this.messagingClient = Suppliers.memoize((Supplier)builder.messagingClient);
        this.instanceIdClient = Suppliers.memoize((Supplier)builder.instanceIdClient);
    }

    public static FirebaseMessaging getInstance() {
        return FirebaseMessaging.getInstance(FirebaseApp.getInstance());
    }

    public static synchronized FirebaseMessaging getInstance(FirebaseApp app) {
        FirebaseMessagingService service = ImplFirebaseTrampolines.getService(app, SERVICE_ID, FirebaseMessagingService.class);
        if (service == null) {
            service = ImplFirebaseTrampolines.addService(app, new FirebaseMessagingService(app));
        }
        return (FirebaseMessaging)service.getInstance();
    }

    public String send(@NonNull Message message) throws FirebaseMessagingException {
        return this.send(message, false);
    }

    public String send(@NonNull Message message, boolean dryRun) throws FirebaseMessagingException {
        return this.sendOp(message, dryRun).call();
    }

    public ApiFuture<String> sendAsync(@NonNull Message message) {
        return this.sendAsync(message, false);
    }

    public ApiFuture<String> sendAsync(@NonNull Message message, boolean dryRun) {
        return this.sendOp(message, dryRun).callAsync(this.app);
    }

    private CallableOperation<String, FirebaseMessagingException> sendOp(final Message message, final boolean dryRun) {
        Preconditions.checkNotNull((Object)message, (Object)"message must not be null");
        final FirebaseMessagingClient messagingClient = this.getMessagingClient();
        return new CallableOperation<String, FirebaseMessagingException>(){

            @Override
            protected String execute() throws FirebaseMessagingException {
                return messagingClient.send(message, dryRun);
            }
        };
    }

    public BatchResponse sendEach(@NonNull List<Message> messages) throws FirebaseMessagingException {
        return this.sendEach(messages, false);
    }

    public BatchResponse sendEach(@NonNull List<Message> messages, boolean dryRun) throws FirebaseMessagingException {
        try {
            return (BatchResponse)this.sendEachOpAsync(messages, dryRun).get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new FirebaseMessagingException(ErrorCode.CANCELLED, SERVICE_ID);
        }
    }

    public ApiFuture<BatchResponse> sendEachAsync(@NonNull List<Message> messages) {
        return this.sendEachOpAsync(messages, false);
    }

    public ApiFuture<BatchResponse> sendEachAsync(@NonNull List<Message> messages, boolean dryRun) {
        return this.sendEachOpAsync(messages, dryRun);
    }

    private ApiFuture<BatchResponse> sendEachOpAsync(List<Message> messages, boolean dryRun) {
        ImmutableList immutableMessages = ImmutableList.copyOf(messages);
        Preconditions.checkArgument((!immutableMessages.isEmpty() ? 1 : 0) != 0, (Object)"messages list must not be empty");
        Preconditions.checkArgument((immutableMessages.size() <= 500 ? 1 : 0) != 0, (Object)"messages list must not contain more than 500 elements");
        ArrayList<ApiFuture<SendResponse>> list = new ArrayList<ApiFuture<SendResponse>>();
        for (Message message : immutableMessages) {
            ApiFuture<SendResponse> messageId = this.sendOpForSendResponse(message, dryRun).callAsync(this.app);
            list.add(messageId);
        }
        ApiFuture responsesFuture = ApiFutures.allAsList(list);
        return ApiFutures.transform((ApiFuture)responsesFuture, responses -> new BatchResponseImpl((List<SendResponse>)responses), (Executor)MoreExecutors.directExecutor());
    }

    private CallableOperation<SendResponse, FirebaseMessagingException> sendOpForSendResponse(final Message message, final boolean dryRun) {
        Preconditions.checkNotNull((Object)message, (Object)"message must not be null");
        final FirebaseMessagingClient messagingClient = this.getMessagingClient();
        return new CallableOperation<SendResponse, FirebaseMessagingException>(){

            @Override
            protected SendResponse execute() {
                try {
                    String messageId = messagingClient.send(message, dryRun);
                    return SendResponse.fromMessageId(messageId);
                }
                catch (FirebaseMessagingException e) {
                    return SendResponse.fromException(e);
                }
            }
        };
    }

    public BatchResponse sendEachForMulticast(@NonNull MulticastMessage message) throws FirebaseMessagingException {
        return this.sendEachForMulticast(message, false);
    }

    public BatchResponse sendEachForMulticast(@NonNull MulticastMessage message, boolean dryRun) throws FirebaseMessagingException {
        Preconditions.checkNotNull((Object)message, (Object)"multicast message must not be null");
        return this.sendEach(message.getMessageList(), dryRun);
    }

    public ApiFuture<BatchResponse> sendEachForMulticastAsync(@NonNull MulticastMessage message) {
        return this.sendEachForMulticastAsync(message, false);
    }

    public ApiFuture<BatchResponse> sendEachForMulticastAsync(@NonNull MulticastMessage message, boolean dryRun) {
        Preconditions.checkNotNull((Object)message, (Object)"multicast message must not be null");
        return this.sendEachAsync(message.getMessageList(), dryRun);
    }

    @Deprecated
    public BatchResponse sendAll(@NonNull List<Message> messages) throws FirebaseMessagingException {
        return this.sendAll(messages, false);
    }

    @Deprecated
    public BatchResponse sendAll(@NonNull List<Message> messages, boolean dryRun) throws FirebaseMessagingException {
        return this.sendAllOp(messages, dryRun).call();
    }

    @Deprecated
    public ApiFuture<BatchResponse> sendAllAsync(@NonNull List<Message> messages) {
        return this.sendAllAsync(messages, false);
    }

    @Deprecated
    public ApiFuture<BatchResponse> sendAllAsync(@NonNull List<Message> messages, boolean dryRun) {
        return this.sendAllOp(messages, dryRun).callAsync(this.app);
    }

    @Deprecated
    public BatchResponse sendMulticast(@NonNull MulticastMessage message) throws FirebaseMessagingException {
        return this.sendMulticast(message, false);
    }

    @Deprecated
    public BatchResponse sendMulticast(@NonNull MulticastMessage message, boolean dryRun) throws FirebaseMessagingException {
        Preconditions.checkNotNull((Object)message, (Object)"multicast message must not be null");
        return this.sendAll(message.getMessageList(), dryRun);
    }

    @Deprecated
    public ApiFuture<BatchResponse> sendMulticastAsync(@NonNull MulticastMessage message) {
        return this.sendMulticastAsync(message, false);
    }

    @Deprecated
    public ApiFuture<BatchResponse> sendMulticastAsync(@NonNull MulticastMessage message, boolean dryRun) {
        Preconditions.checkNotNull((Object)message, (Object)"multicast message must not be null");
        return this.sendAllAsync(message.getMessageList(), dryRun);
    }

    private CallableOperation<BatchResponse, FirebaseMessagingException> sendAllOp(final List<Message> messages, final boolean dryRun) {
        ImmutableList immutableMessages = ImmutableList.copyOf(messages);
        Preconditions.checkArgument((!immutableMessages.isEmpty() ? 1 : 0) != 0, (Object)"messages list must not be empty");
        Preconditions.checkArgument((immutableMessages.size() <= 500 ? 1 : 0) != 0, (Object)"messages list must not contain more than 500 elements");
        final FirebaseMessagingClient messagingClient = this.getMessagingClient();
        return new CallableOperation<BatchResponse, FirebaseMessagingException>(){

            @Override
            protected BatchResponse execute() throws FirebaseMessagingException {
                return messagingClient.sendAll(messages, dryRun);
            }
        };
    }

    @VisibleForTesting
    FirebaseMessagingClient getMessagingClient() {
        return (FirebaseMessagingClient)this.messagingClient.get();
    }

    public TopicManagementResponse subscribeToTopic(@NonNull List<String> registrationTokens, @NonNull String topic) throws FirebaseMessagingException {
        return this.subscribeOp(registrationTokens, topic).call();
    }

    public ApiFuture<TopicManagementResponse> subscribeToTopicAsync(@NonNull List<String> registrationTokens, @NonNull String topic) {
        return this.subscribeOp(registrationTokens, topic).callAsync(this.app);
    }

    private CallableOperation<TopicManagementResponse, FirebaseMessagingException> subscribeOp(final List<String> registrationTokens, final String topic) {
        this.checkRegistrationTokens(registrationTokens);
        this.checkTopic(topic);
        final InstanceIdClient instanceIdClient = this.getInstanceIdClient();
        return new CallableOperation<TopicManagementResponse, FirebaseMessagingException>(){

            @Override
            protected TopicManagementResponse execute() throws FirebaseMessagingException {
                return instanceIdClient.subscribeToTopic(topic, registrationTokens);
            }
        };
    }

    public TopicManagementResponse unsubscribeFromTopic(@NonNull List<String> registrationTokens, @NonNull String topic) throws FirebaseMessagingException {
        return this.unsubscribeOp(registrationTokens, topic).call();
    }

    public ApiFuture<TopicManagementResponse> unsubscribeFromTopicAsync(@NonNull List<String> registrationTokens, @NonNull String topic) {
        return this.unsubscribeOp(registrationTokens, topic).callAsync(this.app);
    }

    private CallableOperation<TopicManagementResponse, FirebaseMessagingException> unsubscribeOp(final List<String> registrationTokens, final String topic) {
        this.checkRegistrationTokens(registrationTokens);
        this.checkTopic(topic);
        final InstanceIdClient instanceIdClient = this.getInstanceIdClient();
        return new CallableOperation<TopicManagementResponse, FirebaseMessagingException>(){

            @Override
            protected TopicManagementResponse execute() throws FirebaseMessagingException {
                return instanceIdClient.unsubscribeFromTopic(topic, registrationTokens);
            }
        };
    }

    @VisibleForTesting
    InstanceIdClient getInstanceIdClient() {
        return (InstanceIdClient)this.instanceIdClient.get();
    }

    private void checkRegistrationTokens(List<String> registrationTokens) {
        Preconditions.checkArgument((registrationTokens != null && !registrationTokens.isEmpty() ? 1 : 0) != 0, (Object)"registrationTokens list must not be null or empty");
        Preconditions.checkArgument((registrationTokens.size() <= 1000 ? 1 : 0) != 0, (Object)"registrationTokens list must not contain more than 1000 elements");
        for (String token : registrationTokens) {
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)token) ? 1 : 0) != 0, (Object)"registration tokens list must not contain null or empty strings");
        }
    }

    private void checkTopic(String topic) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)topic) ? 1 : 0) != 0, (Object)"topic must not be null or empty");
        Preconditions.checkArgument((boolean)topic.matches("^(/topics/)?(private/)?[a-zA-Z0-9-_.~%]+$"), (Object)"invalid topic name");
    }

    private static FirebaseMessaging fromApp(final FirebaseApp app) {
        return FirebaseMessaging.builder().setFirebaseApp(app).setMessagingClient((Supplier<? extends FirebaseMessagingClient>)new Supplier<FirebaseMessagingClient>(){

            public FirebaseMessagingClient get() {
                return FirebaseMessagingClientImpl.fromApp(app);
            }
        }).setInstanceIdClient((Supplier<? extends InstanceIdClient>)new Supplier<InstanceIdClient>(){

            public InstanceIdClientImpl get() {
                return InstanceIdClientImpl.fromApp(app);
            }
        }).build();
    }

    static Builder builder() {
        return new Builder();
    }

    static class Builder {
        private FirebaseApp firebaseApp;
        private Supplier<? extends FirebaseMessagingClient> messagingClient;
        private Supplier<? extends InstanceIdClient> instanceIdClient;

        private Builder() {
        }

        Builder setFirebaseApp(FirebaseApp firebaseApp) {
            this.firebaseApp = firebaseApp;
            return this;
        }

        Builder setMessagingClient(Supplier<? extends FirebaseMessagingClient> messagingClient) {
            this.messagingClient = messagingClient;
            return this;
        }

        Builder setInstanceIdClient(Supplier<? extends InstanceIdClient> instanceIdClient) {
            this.instanceIdClient = instanceIdClient;
            return this;
        }

        FirebaseMessaging build() {
            return new FirebaseMessaging(this);
        }
    }

    private static class FirebaseMessagingService
    extends FirebaseService<FirebaseMessaging> {
        FirebaseMessagingService(FirebaseApp app) {
            super(SERVICE_ID, FirebaseMessaging.fromApp(app));
        }
    }
}

