// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package chromewebstore provides access to the Chrome Web Store API.
//
// For product documentation, see: https://developer.chrome.com/docs/webstore/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/chromewebstore/v2"
//	...
//	ctx := context.Background()
//	chromewebstoreService, err := chromewebstore.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithScopes(chromewebstore.ChromewebstoreReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package chromewebstore // import "google.golang.org/api/chromewebstore/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "chromewebstore:v2"
const apiName = "chromewebstore"
const apiVersion = "v2"
const basePath = "https://chromewebstore.googleapis.com/"
const basePathTemplate = "https://chromewebstore.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://chromewebstore.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, update, or publish your Chrome Web Store extensions, themes,
	// apps, and licences you have access to
	ChromewebstoreScope = "https://www.googleapis.com/auth/chromewebstore"

	// See and download your Chrome Web Store extensions and apps, and see licenses
	// you have access to
	ChromewebstoreReadonlyScope = "https://www.googleapis.com/auth/chromewebstore.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chromewebstore",
		"https://www.googleapis.com/auth/chromewebstore.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Media = NewMediaService(s)
	s.Publishers = NewPublishersService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Media *MediaService

	Publishers *PublishersService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewMediaService(s *Service) *MediaService {
	rs := &MediaService{s: s}
	return rs
}

type MediaService struct {
	s *Service
}

func NewPublishersService(s *Service) *PublishersService {
	rs := &PublishersService{s: s}
	rs.Items = NewPublishersItemsService(s)
	return rs
}

type PublishersService struct {
	s *Service

	Items *PublishersItemsService
}

func NewPublishersItemsService(s *Service) *PublishersItemsService {
	rs := &PublishersItemsService{s: s}
	return rs
}

type PublishersItemsService struct {
	s *Service
}

// Blobstore2Info: Information to read/write to blobstore2.
type Blobstore2Info struct {
	// BlobGeneration: The blob generation id.
	BlobGeneration int64 `json:"blobGeneration,omitempty,string"`
	// BlobId: The blob id, e.g., /blobstore/prod/playground/scotty
	BlobId string `json:"blobId,omitempty"`
	// DownloadReadHandle: Read handle passed from Bigstore -> Scotty for a GCS
	// download. This is a signed, serialized blobstore2.ReadHandle proto which
	// must never be set outside of Bigstore, and is not applicable to non-GCS
	// media downloads.
	DownloadReadHandle string `json:"downloadReadHandle,omitempty"`
	// ReadToken: The blob read token. Needed to read blobs that have not been
	// replicated. Might not be available until the final call.
	ReadToken string `json:"readToken,omitempty"`
	// UploadMetadataContainer: Metadata passed from Blobstore -> Scotty for a new
	// GCS upload. This is a signed, serialized blobstore2.BlobMetadataContainer
	// proto which must never be consumed outside of Bigstore, and is not
	// applicable to non-GCS media uploads.
	UploadMetadataContainer string `json:"uploadMetadataContainer,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BlobGeneration") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BlobGeneration") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Blobstore2Info) MarshalJSON() ([]byte, error) {
	type NoMethod Blobstore2Info
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CancelSubmissionRequest: Request message for CancelSubmission.
type CancelSubmissionRequest struct {
}

// CancelSubmissionResponse: Response message for `CancelSubmission`.
type CancelSubmissionResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// CompositeMedia: A sequence of media data references representing composite
// data. Introduced to support Bigstore composite objects. For details, visit
// http://go/bigstore-composites.
type CompositeMedia struct {
	// BlobRef: Blobstore v1 reference, set if reference_type is BLOBSTORE_REF This
	// should be the byte representation of a blobstore.BlobRef. Since Blobstore is
	// deprecating v1, use blobstore2_info instead. For now, any v2 blob will also
	// be represented in this field as v1 BlobRef.
	BlobRef string `json:"blobRef,omitempty"`
	// Blobstore2Info: Blobstore v2 info, set if reference_type is BLOBSTORE_REF
	// and it refers to a v2 blob.
	Blobstore2Info *Blobstore2Info `json:"blobstore2Info,omitempty"`
	// CosmoBinaryReference: A binary data reference for a media download. Serves
	// as a technology-agnostic binary reference in some Google infrastructure.
	// This value is a serialized storage_cosmo.BinaryReference proto. Storing it
	// as bytes is a hack to get around the fact that the cosmo proto (as well as
	// others it includes) doesn't support JavaScript. This prevents us from
	// including the actual type of this field.
	CosmoBinaryReference string `json:"cosmoBinaryReference,omitempty"`
	// Crc32cHash: crc32.c hash for the payload.
	Crc32cHash int64 `json:"crc32cHash,omitempty"`
	// Inline: Media data, set if reference_type is INLINE
	Inline string `json:"inline,omitempty"`
	// Length: Size of the data, in bytes
	Length int64 `json:"length,omitempty,string"`
	// Md5Hash: MD5 hash for the payload.
	Md5Hash string `json:"md5Hash,omitempty"`
	// ObjectId: Reference to a TI Blob, set if reference_type is BIGSTORE_REF.
	ObjectId *ObjectId `json:"objectId,omitempty"`
	// Path: Path to the data, set if reference_type is PATH
	Path string `json:"path,omitempty"`
	// ReferenceType: Describes what the field reference contains.
	//
	// Possible values:
	//   "PATH" - Reference contains a GFS path or a local path.
	//   "BLOB_REF" - Reference points to a blobstore object. This could be either
	// a v1 blob_ref or a v2 blobstore2_info. Clients should check blobstore2_info
	// first, since v1 is being deprecated.
	//   "INLINE" - Data is included into this proto buffer
	//   "BIGSTORE_REF" - Reference points to a bigstore object
	//   "COSMO_BINARY_REFERENCE" - Indicates the data is stored in
	// cosmo_binary_reference.
	ReferenceType string `json:"referenceType,omitempty"`
	// Sha1Hash: SHA-1 hash for the payload.
	Sha1Hash string `json:"sha1Hash,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BlobRef") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BlobRef") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CompositeMedia) MarshalJSON() ([]byte, error) {
	type NoMethod CompositeMedia
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ContentTypeInfo: Detailed Content-Type information from Scotty. The
// Content-Type of the media will typically be filled in by the header or
// Scotty's best_guess, but this extended information provides the backend with
// more information so that it can make a better decision if needed. This is
// only used on media upload requests from Scotty.
type ContentTypeInfo struct {
	// BestGuess: Scotty's best guess of what the content type of the file is.
	BestGuess string `json:"bestGuess,omitempty"`
	// FromBytes: The content type of the file derived by looking at specific bytes
	// (i.e. "magic bytes") of the actual file.
	FromBytes string `json:"fromBytes,omitempty"`
	// FromFileName: The content type of the file derived from the file extension
	// of the original file name used by the client.
	FromFileName string `json:"fromFileName,omitempty"`
	// FromHeader: The content type of the file as specified in the request
	// headers, multipart headers, or RUPIO start request.
	FromHeader string `json:"fromHeader,omitempty"`
	// FromUrlPath: The content type of the file derived from the file extension of
	// the URL path. The URL path is assumed to represent a file name (which is
	// typically only true for agents that are providing a REST API).
	FromUrlPath string `json:"fromUrlPath,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BestGuess") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BestGuess") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ContentTypeInfo) MarshalJSON() ([]byte, error) {
	type NoMethod ContentTypeInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DeployInfo: Deployment information for a specific release channel. Used in
// requests to update deployment parameters.
type DeployInfo struct {
	// DeployPercentage: Required. The current deploy percentage for the release
	// channel (nonnegative number between 0 and 100).
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployPercentage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployPercentage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeployInfo) MarshalJSON() ([]byte, error) {
	type NoMethod DeployInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DiffChecksumsResponse: Backend response for a Diff get checksums response.
// For details on the Scotty Diff protocol, visit
// http://go/scotty-diff-protocol.
type DiffChecksumsResponse struct {
	// ChecksumsLocation: Exactly one of these fields must be populated. If
	// checksums_location is filled, the server will return the corresponding
	// contents to the user. If object_location is filled, the server will
	// calculate the checksums based on the content there and return that to the
	// user. For details on the format of the checksums, see
	// http://go/scotty-diff-protocol.
	ChecksumsLocation *CompositeMedia `json:"checksumsLocation,omitempty"`
	// ChunkSizeBytes: The chunk size of checksums. Must be a multiple of 256KB.
	ChunkSizeBytes int64 `json:"chunkSizeBytes,omitempty,string"`
	// ObjectLocation: If set, calculate the checksums based on the contents and
	// return them to the caller.
	ObjectLocation *CompositeMedia `json:"objectLocation,omitempty"`
	// ObjectSizeBytes: The total size of the server object.
	ObjectSizeBytes int64 `json:"objectSizeBytes,omitempty,string"`
	// ObjectVersion: The object version of the object the checksums are being
	// returned for.
	ObjectVersion string `json:"objectVersion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChecksumsLocation") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChecksumsLocation") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DiffChecksumsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod DiffChecksumsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DiffDownloadResponse: Backend response for a Diff download response. For
// details on the Scotty Diff protocol, visit http://go/scotty-diff-protocol.
type DiffDownloadResponse struct {
	// ObjectLocation: The original object location.
	ObjectLocation *CompositeMedia `json:"objectLocation,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ObjectLocation") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ObjectLocation") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DiffDownloadResponse) MarshalJSON() ([]byte, error) {
	type NoMethod DiffDownloadResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DiffUploadRequest: A Diff upload request. For details on the Scotty Diff
// protocol, visit http://go/scotty-diff-protocol.
type DiffUploadRequest struct {
	// ChecksumsInfo: The location of the checksums for the new object. Agents must
	// clone the object located here, as the upload server will delete the contents
	// once a response is received. For details on the format of the checksums, see
	// http://go/scotty-diff-protocol.
	ChecksumsInfo *CompositeMedia `json:"checksumsInfo,omitempty"`
	// ObjectInfo: The location of the new object. Agents must clone the object
	// located here, as the upload server will delete the contents once a response
	// is received.
	ObjectInfo *CompositeMedia `json:"objectInfo,omitempty"`
	// ObjectVersion: The object version of the object that is the base version the
	// incoming diff script will be applied to. This field will always be filled
	// in.
	ObjectVersion string `json:"objectVersion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChecksumsInfo") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChecksumsInfo") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DiffUploadRequest) MarshalJSON() ([]byte, error) {
	type NoMethod DiffUploadRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DiffUploadResponse: Backend response for a Diff upload request. For details
// on the Scotty Diff protocol, visit http://go/scotty-diff-protocol.
type DiffUploadResponse struct {
	// ObjectVersion: The object version of the object at the server. Must be
	// included in the end notification response. The version in the end
	// notification response must correspond to the new version of the object that
	// is now stored at the server, after the upload.
	ObjectVersion string `json:"objectVersion,omitempty"`
	// OriginalObject: The location of the original file for a diff upload request.
	// Must be filled in if responding to an upload start notification.
	OriginalObject *CompositeMedia `json:"originalObject,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ObjectVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ObjectVersion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DiffUploadResponse) MarshalJSON() ([]byte, error) {
	type NoMethod DiffUploadResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DiffVersionResponse: Backend response for a Diff get version response. For
// details on the Scotty Diff protocol, visit http://go/scotty-diff-protocol.
type DiffVersionResponse struct {
	// ObjectSizeBytes: The total size of the server object.
	ObjectSizeBytes int64 `json:"objectSizeBytes,omitempty,string"`
	// ObjectVersion: The version of the object stored at the server.
	ObjectVersion string `json:"objectVersion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ObjectSizeBytes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ObjectSizeBytes") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DiffVersionResponse) MarshalJSON() ([]byte, error) {
	type NoMethod DiffVersionResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DistributionChannel: Deployment information for a specific release channel
type DistributionChannel struct {
	// CrxVersion: The extension version provided in the manifest of the uploaded
	// package.
	CrxVersion string `json:"crxVersion,omitempty"`
	// DeployPercentage: The current deploy percentage for the release channel
	// (nonnegative number between 0 and 100).
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CrxVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CrxVersion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DistributionChannel) MarshalJSON() ([]byte, error) {
	type NoMethod DistributionChannel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DownloadParameters: Parameters specific to media downloads.
type DownloadParameters struct {
	// AllowGzipCompression: A boolean to be returned in the response to Scotty.
	// Allows/disallows gzip encoding of the payload content when the server thinks
	// it's advantageous (hence, does not guarantee compression) which allows
	// Scotty to GZip the response to the client.
	AllowGzipCompression bool `json:"allowGzipCompression,omitempty"`
	// IgnoreRange: Determining whether or not Apiary should skip the inclusion of
	// any Content-Range header on its response to Scotty.
	IgnoreRange bool `json:"ignoreRange,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AllowGzipCompression") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowGzipCompression") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DownloadParameters) MarshalJSON() ([]byte, error) {
	type NoMethod DownloadParameters
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FetchItemStatusResponse: Response message for `FetchItemStatus`.
type FetchItemStatusResponse struct {
	// ItemId: Output only. The ID of the item.
	ItemId string `json:"itemId,omitempty"`
	// LastAsyncUploadState: Output only. The state of the last async upload for an
	// item. Only set when there has been an async upload for the item in the past
	// 24 hours.
	//
	// Possible values:
	//   "UPLOAD_STATE_UNSPECIFIED" - The default value.
	//   "SUCCEEDED" - The upload succeeded.
	//   "IN_PROGRESS" - The upload is currently being processed.
	//   "FAILED" - The upload failed.
	//   "NOT_FOUND" - Used as the value of `lastAsyncUploadState` in a
	// `fetchStatus` response indicating that an upload attempt was not found.
	LastAsyncUploadState string `json:"lastAsyncUploadState,omitempty"`
	// Name: The name of the requested item.
	Name string `json:"name,omitempty"`
	// PublicKey: The public key of the item, which may be generated by the store.
	PublicKey string `json:"publicKey,omitempty"`
	// PublishedItemRevisionStatus: Output only. Status of the current published
	// revision of the item. Will be unset if the item is not published.
	PublishedItemRevisionStatus *ItemRevisionStatus `json:"publishedItemRevisionStatus,omitempty"`
	// SubmittedItemRevisionStatus: Status of the item revision submitted to be
	// published. Will be unset if the item has not been submitted for publishing
	// since the last successful publish.
	SubmittedItemRevisionStatus *ItemRevisionStatus `json:"submittedItemRevisionStatus,omitempty"`
	// TakenDown: If true, the item has been taken down for a policy violation.
	// Check the developer dashboard for details.
	TakenDown bool `json:"takenDown,omitempty"`
	// Warned: If true, the item has been warned for a policy violation and will be
	// taken down if not resolved. Check the developer dashboard for details.
	Warned bool `json:"warned,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ItemId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ItemId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FetchItemStatusResponse) MarshalJSON() ([]byte, error) {
	type NoMethod FetchItemStatusResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ItemRevisionStatus: Details on the status of an item revision.
type ItemRevisionStatus struct {
	// DistributionChannels: Details on the package of the item
	DistributionChannels []*DistributionChannel `json:"distributionChannels,omitempty"`
	// State: Output only. Current state of the item
	//
	// Possible values:
	//   "ITEM_STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "PENDING_REVIEW" - The item is pending review.
	//   "STAGED" - The item has been approved and is ready to be published.
	//   "PUBLISHED" - The item is published publicly.
	//   "PUBLISHED_TO_TESTERS" - The item is published to trusted testers.
	//   "REJECTED" - The item has been rejected for publishing.
	//   "CANCELLED" - The item submission has been cancelled.
	State string `json:"state,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DistributionChannels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DistributionChannels") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemRevisionStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ItemRevisionStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Media: A reference to data stored on the filesystem, on GFS or in blobstore.
type Media struct {
	// Algorithm: Deprecated, use one of explicit hash type fields instead.
	// Algorithm used for calculating the hash. As of 2011/01/21, "MD5" is the only
	// possible value for this field. New values may be added at any time.
	Algorithm string `json:"algorithm,omitempty"`
	// BigstoreObjectRef: Use object_id instead.
	BigstoreObjectRef string `json:"bigstoreObjectRef,omitempty"`
	// BlobRef: Blobstore v1 reference, set if reference_type is BLOBSTORE_REF This
	// should be the byte representation of a blobstore.BlobRef. Since Blobstore is
	// deprecating v1, use blobstore2_info instead. For now, any v2 blob will also
	// be represented in this field as v1 BlobRef.
	BlobRef string `json:"blobRef,omitempty"`
	// Blobstore2Info: Blobstore v2 info, set if reference_type is BLOBSTORE_REF
	// and it refers to a v2 blob.
	Blobstore2Info *Blobstore2Info `json:"blobstore2Info,omitempty"`
	// CompositeMedia: A composite media composed of one or more media objects, set
	// if reference_type is COMPOSITE_MEDIA. The media length field must be set to
	// the sum of the lengths of all composite media objects. Note: All composite
	// media must have length specified.
	CompositeMedia []*CompositeMedia `json:"compositeMedia,omitempty"`
	// ContentType: MIME type of the data
	ContentType string `json:"contentType,omitempty"`
	// ContentTypeInfo: Extended content type information provided for Scotty
	// uploads.
	ContentTypeInfo *ContentTypeInfo `json:"contentTypeInfo,omitempty"`
	// CosmoBinaryReference: A binary data reference for a media download. Serves
	// as a technology-agnostic binary reference in some Google infrastructure.
	// This value is a serialized storage_cosmo.BinaryReference proto. Storing it
	// as bytes is a hack to get around the fact that the cosmo proto (as well as
	// others it includes) doesn't support JavaScript. This prevents us from
	// including the actual type of this field.
	CosmoBinaryReference string `json:"cosmoBinaryReference,omitempty"`
	// Crc32cHash: For Scotty Uploads: Scotty-provided hashes for uploads For
	// Scotty Downloads: (WARNING: DO NOT USE WITHOUT PERMISSION FROM THE SCOTTY
	// TEAM.) A Hash provided by the agent to be used to verify the data being
	// downloaded. Currently only supported for inline payloads. Further, only
	// crc32c_hash is currently supported.
	Crc32cHash int64 `json:"crc32cHash,omitempty"`
	// DiffChecksumsResponse: Set if reference_type is DIFF_CHECKSUMS_RESPONSE.
	DiffChecksumsResponse *DiffChecksumsResponse `json:"diffChecksumsResponse,omitempty"`
	// DiffDownloadResponse: Set if reference_type is DIFF_DOWNLOAD_RESPONSE.
	DiffDownloadResponse *DiffDownloadResponse `json:"diffDownloadResponse,omitempty"`
	// DiffUploadRequest: Set if reference_type is DIFF_UPLOAD_REQUEST.
	DiffUploadRequest *DiffUploadRequest `json:"diffUploadRequest,omitempty"`
	// DiffUploadResponse: Set if reference_type is DIFF_UPLOAD_RESPONSE.
	DiffUploadResponse *DiffUploadResponse `json:"diffUploadResponse,omitempty"`
	// DiffVersionResponse: Set if reference_type is DIFF_VERSION_RESPONSE.
	DiffVersionResponse *DiffVersionResponse `json:"diffVersionResponse,omitempty"`
	// DownloadParameters: Parameters for a media download.
	DownloadParameters *DownloadParameters `json:"downloadParameters,omitempty"`
	// Filename: Original file name
	Filename string `json:"filename,omitempty"`
	// Hash: Deprecated, use one of explicit hash type fields instead. These two
	// hash related fields will only be populated on Scotty based media uploads and
	// will contain the content of the hash group in the NotificationRequest:
	// http://cs/#google3/blobstore2/api/scotty/service/proto/upload_listener.proto&q=class:Hash
	// Hex encoded hash value of the uploaded media.
	Hash string `json:"hash,omitempty"`
	// HashVerified: For Scotty uploads only. If a user sends a hash code and the
	// backend has requested that Scotty verify the upload against the client hash,
	// Scotty will perform the check on behalf of the backend and will reject it if
	// the hashes don't match. This is set to true if Scotty performed this
	// verification.
	HashVerified bool `json:"hashVerified,omitempty"`
	// Inline: Media data, set if reference_type is INLINE
	Inline string `json:"inline,omitempty"`
	// IsPotentialRetry: |is_potential_retry| is set false only when Scotty is
	// certain that it has not sent the request before. When a client resumes an
	// upload, this field must be set true in agent calls, because Scotty cannot be
	// certain that it has never sent the request before due to potential failure
	// in the session state persistence.
	IsPotentialRetry bool `json:"isPotentialRetry,omitempty"`
	// Length: Size of the data, in bytes
	Length int64 `json:"length,omitempty,string"`
	// Md5Hash: Scotty-provided MD5 hash for an upload.
	Md5Hash string `json:"md5Hash,omitempty"`
	// MediaId: Media id to forward to the operation GetMedia. Can be set if
	// reference_type is GET_MEDIA.
	MediaId string `json:"mediaId,omitempty"`
	// ObjectId: Reference to a TI Blob, set if reference_type is BIGSTORE_REF.
	ObjectId *ObjectId `json:"objectId,omitempty"`
	// Path: Path to the data, set if reference_type is PATH
	Path string `json:"path,omitempty"`
	// ReferenceType: Describes what the field reference contains.
	//
	// Possible values:
	//   "PATH" - Reference contains a GFS path or a local path.
	//   "BLOB_REF" - Reference points to a blobstore object. This could be either
	// a v1 blob_ref or a v2 blobstore2_info. Clients should check blobstore2_info
	// first, since v1 is being deprecated.
	//   "INLINE" - Data is included into this proto buffer
	//   "GET_MEDIA" - Data should be accessed from the current service using the
	// operation GetMedia.
	//   "COMPOSITE_MEDIA" - The content for this media object is stored across
	// multiple partial media objects under the composite_media field.
	//   "BIGSTORE_REF" - Reference points to a bigstore object
	//   "DIFF_VERSION_RESPONSE" - Indicates the data is stored in
	// diff_version_response.
	//   "DIFF_CHECKSUMS_RESPONSE" - Indicates the data is stored in
	// diff_checksums_response.
	//   "DIFF_DOWNLOAD_RESPONSE" - Indicates the data is stored in
	// diff_download_response.
	//   "DIFF_UPLOAD_REQUEST" - Indicates the data is stored in
	// diff_upload_request.
	//   "DIFF_UPLOAD_RESPONSE" - Indicates the data is stored in
	// diff_upload_response.
	//   "COSMO_BINARY_REFERENCE" - Indicates the data is stored in
	// cosmo_binary_reference.
	//   "ARBITRARY_BYTES" - Informs Scotty to generate a response payload with the
	// size specified in the length field. The contents of the payload are
	// generated by Scotty and are undefined. This is useful for testing download
	// speeds between the user and Scotty without involving a real payload source.
	// Note: range is not supported when using arbitrary_bytes.
	ReferenceType string `json:"referenceType,omitempty"`
	// Sha1Hash: Scotty-provided SHA1 hash for an upload.
	Sha1Hash string `json:"sha1Hash,omitempty"`
	// Sha256Hash: Scotty-provided SHA256 hash for an upload.
	Sha256Hash string `json:"sha256Hash,omitempty"`
	// Timestamp: Time at which the media data was last updated, in milliseconds
	// since UNIX epoch
	Timestamp uint64 `json:"timestamp,omitempty,string"`
	// Token: A unique fingerprint/version id for the media data
	Token string `json:"token,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Algorithm") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Algorithm") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Media) MarshalJSON() ([]byte, error) {
	type NoMethod Media
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MediaRequestInfo: Extra information added to operations that support Scotty
// media requests.
type MediaRequestInfo struct {
	// CurrentBytes: The number of current bytes uploaded or downloaded.
	CurrentBytes int64 `json:"currentBytes,omitempty,string"`
	// CustomData: Data to be copied to backend requests. Custom data is returned
	// to Scotty in the agent_state field, which Scotty will then provide in
	// subsequent upload notifications.
	CustomData string `json:"customData,omitempty"`
	// DiffObjectVersion: Set if the http request info is diff encoded. The value
	// of this field is the version number of the base revision. This is
	// corresponding to Apiary's mediaDiffObjectVersion
	// (//depot/google3/java/com/google/api/server/media/variable/DiffObjectVersionV
	// ariable.java). See go/esf-scotty-diff-upload for more information.
	DiffObjectVersion string `json:"diffObjectVersion,omitempty"`
	// FinalStatus: The existence of the final_status field indicates that this is
	// the last call to the agent for this request_id.
	// http://google3/uploader/agent/scotty_agent.proto?l=737&rcl=347601929
	FinalStatus int64 `json:"finalStatus,omitempty"`
	// NotificationType: The type of notification received from Scotty.
	//
	// Possible values:
	//   "START" - Such requests signals the start of a request containing media
	// upload. Only the media field(s) in the inserted/updated resource are set.
	// The response should either return an error or succeed. On success, responses
	// don't need to contain anything.
	//   "PROGRESS" - Such requests signals that the upload has progressed and that
	// the backend might want to access the media file specified in relevant fields
	// in the resource. Only the media field(s) in the inserted/updated resource
	// are set. The response should either return an error or succeed. On success,
	// responses don't need to contain anything.
	//   "END" - Such requests signals the end of a request containing media
	// upload. END should be handled just like normal Insert/Upload requests, that
	// is, they should process the request and return a complete resource in the
	// response. Pointers to media data (a GFS path usually) appear in the relevant
	// fields in the inserted/updated resource. See gdata.Media in data.proto.
	//   "RESPONSE_SENT" - Such requests occur after an END and signal that the
	// response has been sent back to the client. RESPONSE_SENT is only sent to the
	// backend if it is configured to receive them. The response does not need to
	// contain anything.
	//   "ERROR" - Such requests indicate that an error occurred while processing
	// the request. ERROR is only sent to the backend if it is configured to
	// receive them. It is not guaranteed that all errors will result in this
	// notification to the backend, even if the backend requests them. Since these
	// requests are just for informational purposes, the response does not need to
	// contain anything.
	NotificationType string `json:"notificationType,omitempty"`
	// PhysicalHeaders: The physical headers provided by RequestReceivedParameters
	// in Scotty request. type is uploader_service.KeyValuePairs.
	PhysicalHeaders string `json:"physicalHeaders,omitempty"`
	// RequestId: The Scotty request ID.
	RequestId string `json:"requestId,omitempty"`
	// RequestReceivedParamsServingInfo: The partition of the Scotty server
	// handling this request. type is
	// uploader_service.RequestReceivedParamsServingInfo
	// LINT.IfChange(request_received_params_serving_info_annotations)
	// LINT.ThenChange()
	RequestReceivedParamsServingInfo string `json:"requestReceivedParamsServingInfo,omitempty"`
	// TotalBytes: The total size of the file.
	TotalBytes int64 `json:"totalBytes,omitempty,string"`
	// TotalBytesIsEstimated: Whether the total bytes field contains an estimated
	// data.
	TotalBytesIsEstimated bool `json:"totalBytesIsEstimated,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CurrentBytes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CurrentBytes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MediaRequestInfo) MarshalJSON() ([]byte, error) {
	type NoMethod MediaRequestInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ObjectId: This is a copy of the tech.blob.ObjectId proto, which could not be
// used directly here due to transitive closure issues with JavaScript support;
// see http://b/8801763.
type ObjectId struct {
	// BucketName: The name of the bucket to which this object belongs.
	BucketName string `json:"bucketName,omitempty"`
	// Generation: Generation of the object. Generations are monotonically
	// increasing across writes, allowing them to be be compared to determine which
	// generation is newer. If this is omitted in a request, then you are
	// requesting the live object. See http://go/bigstore-versions
	Generation int64 `json:"generation,omitempty,string"`
	// ObjectName: The name of the object.
	ObjectName string `json:"objectName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BucketName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BucketName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ObjectId) MarshalJSON() ([]byte, error) {
	type NoMethod ObjectId
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PublishItemRequest: Request message for PublishItem.
type PublishItemRequest struct {
	// DeployInfos: Optional. Additional deploy information including the desired
	// initial percentage rollout. Defaults to the current value saved in the
	// developer dashboard if unset.
	DeployInfos []*DeployInfo `json:"deployInfos,omitempty"`
	// PublishType: Optional. Use this to control if the item is published
	// immediately on approval or staged for publishing in the future. Defaults to
	// `DEFAULT_PUBLISH` if unset.
	//
	// Possible values:
	//   "PUBLISH_TYPE_UNSPECIFIED" - Default value. This is the same as
	// DEFAULT_PUBLISH.
	//   "DEFAULT_PUBLISH" - The submission will be published immediately after
	// being approved.
	//   "STAGED_PUBLISH" - After approval the submission will be staged and can
	// then be published by the developer.
	PublishType string `json:"publishType,omitempty"`
	// SkipReview: Optional. Whether to attempt to skip item review. The API will
	// validate if the item qualifies and return a validation error if the item
	// requires review. Defaults to `false` if unset.
	SkipReview bool `json:"skipReview,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployInfos") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployInfos") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PublishItemRequest) MarshalJSON() ([]byte, error) {
	type NoMethod PublishItemRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PublishItemResponse: Response message for `PublishItem`.
type PublishItemResponse struct {
	// ItemId: Output only. The ID of the item.
	ItemId string `json:"itemId,omitempty"`
	// Name: The name of the item that was submitted
	Name string `json:"name,omitempty"`
	// State: Output only. The current state of the submission.
	//
	// Possible values:
	//   "ITEM_STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "PENDING_REVIEW" - The item is pending review.
	//   "STAGED" - The item has been approved and is ready to be published.
	//   "PUBLISHED" - The item is published publicly.
	//   "PUBLISHED_TO_TESTERS" - The item is published to trusted testers.
	//   "REJECTED" - The item has been rejected for publishing.
	//   "CANCELLED" - The item submission has been cancelled.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ItemId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ItemId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PublishItemResponse) MarshalJSON() ([]byte, error) {
	type NoMethod PublishItemResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetPublishedDeployPercentageRequest: Request message for
// SetPublishedDeployPercentage.
type SetPublishedDeployPercentageRequest struct {
	// DeployPercentage: Required. Unscaled percentage value for the publised
	// revision (nonnegative number between 0 and 100). It must be larger than the
	// existing target percentage.
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployPercentage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployPercentage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SetPublishedDeployPercentageRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetPublishedDeployPercentageRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetPublishedDeployPercentageResponse: Response message for
// `SetPublishedDeployPercentage`.
type SetPublishedDeployPercentageResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// UploadItemPackageRequest: Request message for UploadItemPackage.
type UploadItemPackageRequest struct {
	// Blob: Reference to the uploaded media.
	Blob *Media `json:"blob,omitempty"`
	// MediaRequestInfo: Info about the media upload request.
	MediaRequestInfo *MediaRequestInfo `json:"mediaRequestInfo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Blob") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Blob") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UploadItemPackageRequest) MarshalJSON() ([]byte, error) {
	type NoMethod UploadItemPackageRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UploadItemPackageResponse: Response message for `UploadItemPackage`.
type UploadItemPackageResponse struct {
	// CrxVersion: The extension version provided in the manifest of the uploaded
	// package. This will not be set if the upload is still in progress
	// (`upload_state` is `UPLOAD_IN_PROGRESS`).
	CrxVersion string `json:"crxVersion,omitempty"`
	// ItemId: Output only. The ID of the item the package was uploaded to.
	ItemId string `json:"itemId,omitempty"`
	// Name: The name of the item the package was uploaded to.
	Name string `json:"name,omitempty"`
	// UploadState: Output only. The state of the upload. If `upload_state` is
	// `UPLOAD_IN_PROGRESS`, you can poll for updates using the fetchStatus method.
	//
	// Possible values:
	//   "UPLOAD_STATE_UNSPECIFIED" - The default value.
	//   "SUCCEEDED" - The upload succeeded.
	//   "IN_PROGRESS" - The upload is currently being processed.
	//   "FAILED" - The upload failed.
	//   "NOT_FOUND" - Used as the value of `lastAsyncUploadState` in a
	// `fetchStatus` response indicating that an upload attempt was not found.
	UploadState string `json:"uploadState,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CrxVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CrxVersion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UploadItemPackageResponse) MarshalJSON() ([]byte, error) {
	type NoMethod UploadItemPackageResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type MediaUploadCall struct {
	s                        *Service
	name                     string
	uploaditempackagerequest *UploadItemPackageRequest
	urlParams_               gensupport.URLParams
	mediaInfo_               *gensupport.MediaInfo
	ctx_                     context.Context
	header_                  http.Header
}

// Upload: Upload a new package to an existing item.
//
//   - name: Name of the item to upload the new package to in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *MediaService) Upload(name string, uploaditempackagerequest *UploadItemPackageRequest) *MediaUploadCall {
	c := &MediaUploadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.uploaditempackagerequest = uploaditempackagerequest
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *MediaUploadCall) Media(r io.Reader, options ...googleapi.MediaOption) *MediaUploadCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *MediaUploadCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *MediaUploadCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *MediaUploadCall) ProgressUpdater(pu googleapi.ProgressUpdater) *MediaUploadCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MediaUploadCall) Fields(s ...googleapi.Field) *MediaUploadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *MediaUploadCall) Context(ctx context.Context) *MediaUploadCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MediaUploadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaUploadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.uploaditempackagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:upload")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/v2/{+name}:upload")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.media.upload", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.media.upload" call.
// Any non-2xx status code is an error. Response headers are in either
// *UploadItemPackageResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *MediaUploadCall) Do(opts ...googleapi.CallOption) (*UploadItemPackageResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &UploadItemPackageResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.media.upload", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsCancelSubmissionCall struct {
	s                       *Service
	name                    string
	cancelsubmissionrequest *CancelSubmissionRequest
	urlParams_              gensupport.URLParams
	ctx_                    context.Context
	header_                 http.Header
}

// CancelSubmission: Cancel the current active submission of an item if
// present. This can be used to cancel the review of a pending submission.
//
//   - name: Name of the item to cancel the submission of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) CancelSubmission(name string, cancelsubmissionrequest *CancelSubmissionRequest) *PublishersItemsCancelSubmissionCall {
	c := &PublishersItemsCancelSubmissionCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.cancelsubmissionrequest = cancelsubmissionrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsCancelSubmissionCall) Fields(s ...googleapi.Field) *PublishersItemsCancelSubmissionCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsCancelSubmissionCall) Context(ctx context.Context) *PublishersItemsCancelSubmissionCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsCancelSubmissionCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsCancelSubmissionCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.cancelsubmissionrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:cancelSubmission")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.cancelSubmission", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.cancelSubmission" call.
// Any non-2xx status code is an error. Response headers are in either
// *CancelSubmissionResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsCancelSubmissionCall) Do(opts ...googleapi.CallOption) (*CancelSubmissionResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CancelSubmissionResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.cancelSubmission", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsFetchStatusCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// FetchStatus: Fetch the status of an item.
//
//   - name: Name of the item to retrieve the status of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) FetchStatus(name string) *PublishersItemsFetchStatusCall {
	c := &PublishersItemsFetchStatusCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsFetchStatusCall) Fields(s ...googleapi.Field) *PublishersItemsFetchStatusCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PublishersItemsFetchStatusCall) IfNoneMatch(entityTag string) *PublishersItemsFetchStatusCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsFetchStatusCall) Context(ctx context.Context) *PublishersItemsFetchStatusCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsFetchStatusCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsFetchStatusCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:fetchStatus")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.fetchStatus", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.fetchStatus" call.
// Any non-2xx status code is an error. Response headers are in either
// *FetchItemStatusResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsFetchStatusCall) Do(opts ...googleapi.CallOption) (*FetchItemStatusResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &FetchItemStatusResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.fetchStatus", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsPublishCall struct {
	s                  *Service
	name               string
	publishitemrequest *PublishItemRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Publish: Submit the item to be published in the store. The item will be
// submitted for review unless `skip_review` is set to true, or the item is
// staged from a previous submission with `publish_type` set to
// `STAGED_PUBLISH`.
//
//   - name: Name of the item in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) Publish(name string, publishitemrequest *PublishItemRequest) *PublishersItemsPublishCall {
	c := &PublishersItemsPublishCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.publishitemrequest = publishitemrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsPublishCall) Fields(s ...googleapi.Field) *PublishersItemsPublishCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsPublishCall) Context(ctx context.Context) *PublishersItemsPublishCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsPublishCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsPublishCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.publishitemrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:publish")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.publish", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.publish" call.
// Any non-2xx status code is an error. Response headers are in either
// *PublishItemResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *PublishersItemsPublishCall) Do(opts ...googleapi.CallOption) (*PublishItemResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PublishItemResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.publish", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsSetPublishedDeployPercentageCall struct {
	s                                   *Service
	name                                string
	setpublisheddeploypercentagerequest *SetPublishedDeployPercentageRequest
	urlParams_                          gensupport.URLParams
	ctx_                                context.Context
	header_                             http.Header
}

// SetPublishedDeployPercentage: Set a higher target deploy percentage for the
// item's published revision. This will be updated without the item being
// submitted for review. This is only available to items with over 10,000
// seven-day active users.
//
//   - name: Name of the item to update the published revision of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) SetPublishedDeployPercentage(name string, setpublisheddeploypercentagerequest *SetPublishedDeployPercentageRequest) *PublishersItemsSetPublishedDeployPercentageCall {
	c := &PublishersItemsSetPublishedDeployPercentageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.setpublisheddeploypercentagerequest = setpublisheddeploypercentagerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Fields(s ...googleapi.Field) *PublishersItemsSetPublishedDeployPercentageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Context(ctx context.Context) *PublishersItemsSetPublishedDeployPercentageCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsSetPublishedDeployPercentageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.setpublisheddeploypercentagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:setPublishedDeployPercentage")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.setPublishedDeployPercentage", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.setPublishedDeployPercentage" call.
// Any non-2xx status code is an error. Response headers are in either
// *SetPublishedDeployPercentageResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Do(opts ...googleapi.CallOption) (*SetPublishedDeployPercentageResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SetPublishedDeployPercentageResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.setPublishedDeployPercentage", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
